---
jupyter:
  jupytext:
    encoding: '# -*- coding: utf-8 -*-'
    formats: ipynb,py
    text_representation:
      extension: .md
      format_name: markdown
      format_version: '1.2'
      jupytext_version: 1.6.0
  kernelspec:
    display_name: Python 3
    language: python
    name: python3
---

# Probando con TSP


## Primero cargamos las librerías

- tsplib95 para leer ejemplos.
- networkx para obtenerlo en estructura de grafo.
- matplotlib para visualizarlo

```python
import tsplib95
import networkx as nx
import matplotlib.pyplot as plt
from GA import optimize
```

Cargo un fichero

```python
from os import listdir
files = listdir("data")
```

```python
prob = tsplib95.load_problem(f"data/gr120.tsp")
```

## Datos de un problema


Se puede obtener fácilmente las características de un problema

```python
print(prob.comment)
print(prob.dimension)
```

Se pueden recuperar los nodos y conexiones mediante iteradores


con wfun se puede obtener la distancia

```python
print(prob.wfunc(1,1), prob.wfunc(1,50))
```

## Trabajando como un Grafo

```python
G = prob.get_graph()
```

Las opciones se pueden indicar con https://networkx.github.io/documentation/stable/reference/generated/networkx.drawing.nx_pylab.draw_networkx.html#networkx.drawing.nx_pylab.draw_networkx

```python
def scale_plot_size(factor=1.5):
    import matplotlib as mpl
    default_dpi = mpl.rcParamsDefault['figure.dpi']
    mpl.rcParams['figure.dpi'] = default_dpi*factor
```

```python
scale_plot_size(2)
nx.draw_networkx(G, with_labels=False, node_size=10, edgelist=[])
```

La idea sería poner los "edge" de las soluciones

```python
from itertools import permutations, islice
# Get all permutations of [1, 2, 3] 
perm = permutations(list(prob.get_nodes()))
```

```python
def get_edge(sol):
    edge = [(sol[i],sol[i+1]) for i in range(len(sol)-1)]
    edge.append((sol[-1], sol[0]))
    return edge
```

```python
for x in islice(perm, 10, 11):
    edges=get_edge(x)
    nx.draw_networkx(G, with_labels=False, node_size=10, edgelist=edges, width=0.5)
    break
```

# Animación 

Sólo faltaría animarlo, ver algún ejemplo

```python
from matplotlib.animation import FuncAnimation
import numpy as np
from IPython.display import HTML

pos = nx.spring_layout(G)

def update_func(step, sol):
    pass

fig, ax = plt.subplots()
ndim = prob.dimension
        
def update(n):
    ax.clear()
    sol = np.random.permutation(ndim)+1
    edges_sol = get_edge(sol)
    nodes = nx.draw_networkx_nodes(G, pos, with_labels=False, node_size=20)
    edges = nx.draw_networkx_edges(G, pos, edgelist=edges_sol, width=0.5)    
    # edges_sol = get_edge(sol)[:10]
    # print(edges_sol)
    # edges.set_array(edges_sol)
    ax.set_title(f"Frame {n}") 
    return nodes, edges

anim = FuncAnimation(fig, update, interval=1, frames=30, blit=True)
HTML(anim.to_jshtml())
# plt.show()
```

# Aplicando algoritmo Genético


Creamos la función de evaluación

```python
def feval(sol):
    dist = 0
    ndim = len(sol)
    
    for i in range(ndim-1):
        dist += prob.wfunc(sol[i], sol[i+1])
        
    dist += prob.wfunc(sol[-1], sol[0])
    return dist
```

```python
global_bests = []
global_fits = []

def callback(itera, pop, fits):
    if itera % 50 == 1:
        best = fits.argmin()
        # Copy is needed
        global_bests.append(np.copy(pop[best,:]))
        global_fits.append(fits[best])
    # print(f"Iteración[{itera}]: {fits.max()} - {fits.min()}")
```

```python
def apply_GA():
    ndim = prob.dimension
    popsize = 50
    optimize(feval, ndim*100, popsize=popsize, ndim=ndim, callback=callback)
```

```python
global_bests = []
global_fits = []
apply_GA()
```

```python
fig, ax = plt.subplots()
ndim = prob.dimension
import matplotlib
matplotlib.rcParams['animation.embed_limit'] = 2**128
        
def update(n):
    ax.clear()
    sol = global_bests[n]
    edges_sol = get_edge(sol)
    nodes = nx.draw_networkx_nodes(G, pos, with_labels=False, node_size=20)
    edges = nx.draw_networkx_edges(G, pos, edgelist=edges_sol, width=0.5)    
    # edges_sol = get_edge(sol)[:10]
    # print(edges_sol)
    # edges.set_array(edges_sol)
    ax.set_title(f"Frame {n}: Best: {global_fits[n]}") 
    return nodes, edges

anim = FuncAnimation(fig, update, interval=1, frames=min(len(global_fits), 500), blit=True)
HTML(anim.to_jshtml())
```
```python

```

